﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Net.Http.Formatting;
using System.Net.Http.Headers;
using Vimt.JsonWebToken.Messages.Messages;
using VIMT.VideoVisit.Services.Configuration;
using VIMT.VideoVisit.Services.XSD;
using VRM.Integration.Servicebus.Core;
using Logger = VRM.Integration.Servicebus.Core.Logger;

namespace VIMT.VideoVisit.Services.Rest
{
    public static class ServiceFactory
    {
        public static WriteResults CreateAppointment(Appointment payload, string samlToken)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

            var validationResult = AppointmentValidator.ValidateCreateAppointment(payload);

            if (!string.IsNullOrEmpty(validationResult))
            {
                Logger.Instance.Error(string.Format("VVS CreateAppointment Validation Error: {0}", validationResult));
                throw new Exception(validationResult);
            }

            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(VideoVisitSecurityConfiguration.Current.BaseUri);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/xml"));
                client.DefaultRequestHeaders.Referrer = new Uri(VideoVisitSecurityConfiguration.Current.RefererUri);

                try
                {
                    var tokenRequest = new VimtJwtEncryptTokenRequest { SamlToken = samlToken };
                    var tokenResponse = tokenRequest.SendReceive<VimtJwtEncryptTokenResponse>(MessageProcessType.Local);
                    if (tokenResponse != null && tokenResponse.ExceptionOccured)
                    {
                        Logger.Instance.Error(string.Format("VVS CreateAppointment. Error getting encrypted JWT token: {0}", tokenResponse.ExceptionMessage));
                        throw new Exception(tokenResponse.ExceptionMessage);
                    }

                    if (tokenResponse == null)
                    {
                        Logger.Instance.Error("VVS CreateAppointment. JWT token response is null");
                        throw new Exception("VVS CreateAppointment. JWT token response is null");
                    }

                    client.DefaultRequestHeaders.Add("X-VAMF-JWT", tokenResponse.EncryptedJwtToken);
                    var response = client.PostAsync(VideoVisitSecurityConfiguration.Current.CreateUri, payload, new XmlMediaTypeFormatter { UseXmlSerializer = true }).GetAwaiter().GetResult();
                    Logger.Instance.Debug(string.Format("VVS CreateAppointment Status: {0} Reason: {1}", response.StatusCode, response.ReasonPhrase));

                    return HandleHttpResponse(response, "VVS CreateAppointment");
                }
                catch (Exception ex)
                {
                    Logger.Instance.Error(string.Format("Failed VVS CreateAppointment. Reason : {0}", ex.Message));
                    throw new Exception(string.Format("Failed VVS CreateAppointment. Reason : {0}", ex.Message), ex.InnerException);
                }
            }
        }

        public static WriteResults UpdateAppointment(Appointment payload, string samlToken)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

            var validationResult = AppointmentValidator.ValidateCreateAppointment(payload);

            if (!string.IsNullOrEmpty(validationResult))
            {
                Logger.Instance.Error(string.Format("VVS UpdateAppointment Validation Error: {0}", validationResult));
                throw new Exception(validationResult);
            }

            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(VideoVisitSecurityConfiguration.Current.BaseUri);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/xml"));
                client.DefaultRequestHeaders.Referrer = new Uri(VideoVisitSecurityConfiguration.Current.RefererUri);

                try
                {
                    var tokenRequest = new VimtJwtEncryptTokenRequest { SamlToken = samlToken };
                    var tokenResponse = tokenRequest.SendReceive<VimtJwtEncryptTokenResponse>(MessageProcessType.Local);
                    if (tokenResponse != null && tokenResponse.ExceptionOccured)
                    {
                        Logger.Instance.Error(string.Format("VVS UpdateAppointment. Error getting encrypted JWT token: {0}", tokenResponse.ExceptionMessage));
                        throw new Exception(tokenResponse.ExceptionMessage);
                    }

                    if (tokenResponse == null)
                    {
                        Logger.Instance.Error("VVS UpdateAppointment. JWT token response is null");
                        throw new Exception("VVS UpdateAppointment. JWT token response is null");
                    }

                    client.DefaultRequestHeaders.Add("X-VAMF-JWT", tokenResponse.EncryptedJwtToken);
                    var updateUri = string.Format(VideoVisitSecurityConfiguration.Current.UpdateUri, payload.Id);
                    var response = client.PutAsync(updateUri, payload, new XmlMediaTypeFormatter { UseXmlSerializer = true }).GetAwaiter().GetResult();
                    Logger.Instance.Debug(string.Format("VVS UpdateAppointment Status: {0} Reason: {1}", response.StatusCode, response.ReasonPhrase));

                    return HandleHttpResponse(response, "VVS UpdateAppointment");
                }
                catch (Exception ex)
                {
                    Logger.Instance.Error(string.Format("Failed VVS UpdateAppointment. Reason : {0}", ex.Message));
                    throw new Exception(string.Format("Failed VVS UpdateAppointment. Reason : {0}", ex.Message), ex.InnerException);
                }
            }
        }

        public static WriteResults CancelAppointment(CancelAppointmentRequest payload, string samlToken)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

            var validationResult = AppointmentValidator.ValidateCancelAppointment(payload);

            if (!string.IsNullOrEmpty(validationResult))
            {
                Logger.Instance.Error(string.Format("VVS CancelAppointment Validation Error: {0}", validationResult));
                throw new Exception(validationResult);
            }

            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(VideoVisitSecurityConfiguration.Current.BaseUri);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/xml"));
                client.DefaultRequestHeaders.Referrer = new Uri(VideoVisitSecurityConfiguration.Current.RefererUri);

                try
                {
                    var tokenRequest = new VimtJwtEncryptTokenRequest { SamlToken = samlToken };
                    var tokenResponse = tokenRequest.SendReceive<VimtJwtEncryptTokenResponse>(MessageProcessType.Local);
                    if (tokenResponse != null && tokenResponse.ExceptionOccured)
                    {
                        Logger.Instance.Error(string.Format("VVS CancelAppointment. Error getting encrypted JWT token: {0}", tokenResponse.ExceptionMessage));
                        throw new Exception(tokenResponse.ExceptionMessage);
                    }

                    if (tokenResponse == null)
                    {
                        Logger.Instance.Error("VVS CancelAppointment. JWT token response is null");
                        throw new Exception("VVS CancelAppointment. JWT token response is null");
                    }

                    client.DefaultRequestHeaders.Add("X-VAMF-JWT", tokenResponse.EncryptedJwtToken);
                    var cancelUri = string.Format(VideoVisitSecurityConfiguration.Current.CancelUri, payload.Id);
                    var response = client.PutAsync(cancelUri, payload, new XmlMediaTypeFormatter { UseXmlSerializer = true }).GetAwaiter().GetResult();
                    Logger.Instance.Debug(string.Format("VVS CancelAppointment Status: {0} Reason: {1}", response.StatusCode, response.ReasonPhrase));

                    return HandleHttpResponse(response, "VVS CancelAppointment");
                }
                catch (Exception ex)
                {
                    Logger.Instance.Error(string.Format("Failed VVS CancelAppointment. Reason : {0}", ex.Message));
                    throw new Exception(string.Format("Failed VVS CancelAppointment. Reason : {0}", ex.Message), ex.InnerException);
                }
            }
        }

        public static WriteResults HandleHttpResponse(HttpResponseMessage response, string reqType)
        {
            if (response.IsSuccessStatusCode)
            {
                Logger.Instance.Debug(string.Format("{0} was successfully sent", reqType));
                var apptResponse = response.Content.ReadAsAsync<AppointmentResponse>(new List<MediaTypeFormatter> { new XmlMediaTypeFormatter { UseXmlSerializer = true } }).GetAwaiter().GetResult();
                return apptResponse?.WriteResults;
            }

            if (response.StatusCode == HttpStatusCode.BadRequest)
            {
                var errors = response.Content.ReadAsAsync<ValidationErrors>(new List<MediaTypeFormatter>{ new XmlMediaTypeFormatter { UseXmlSerializer = true } }).GetAwaiter().GetResult();
                var errorString = "Validation Error(s):";
                errors.Errors.ForEach(e => errorString += string.Format("\nField Name={0} | Error={1}", e.FieldName, e.ErrorMessage));

                Logger.Instance.Error(string.Format("Validation Errors for {0}. Reason: {1}. The following validation errors were found: {2}", reqType, response.StatusCode, errorString));
                throw new Exception(string.Format("Validation Errors for {0}. Reason: {1}. The following validation errors were found: {2}", reqType, response.StatusCode, errorString));
            }

            var unknownErrorString = response.Content.ReadAsStringAsync().GetAwaiter().GetResult();
            Logger.Instance.Error(string.Format("Unknown Error in {0}. Reason={1} | Unknown Error={2}.", reqType, response.StatusCode, unknownErrorString));
            throw new Exception(string.Format("Unknown Error in {0}. Reason={1} | Unknown Error={2}.", reqType, response.StatusCode, unknownErrorString));
        }
    }
}